/************************************************************************
 * \file: trace_q.c
 *
 * \version: $Id: trace_q.c,v 1.14 2012/05/28 10:30:38 jayanth.mc Exp $
 *
 * This file implements circular queue for Trace.
 *
 * \component: Gen2 Trace
 *
 * \author B. Das         bhaskar.das(o)in.bosch.com
 *         Sakthivelu S.  sakthivelu.s(o)in.bosch.com
 *\Modified: Arun V
 * \copyright: (c) 2003 - 2009 ADIT
 *
 ***********************************************************************/

 #include "trace_q.h"
 #include "trace_base.h"
 

/**
 * Push user passed packet into queue.
 *
 * \param q          Pointer to queue
 * \parm  appl_buf   Pointer to user passed buffer
 *
 * return \li E_OK     On successful completion
 *        \li E_DISWAI In case of a shutdown request
 *        \li          else error returned by concrete implementation.
 */
EXPORT ER TRACE_q_push(TRACE_q* q, VP appl_buf)
{
	ER				rc_qlock	= E_OK;
	ER				ret_val		= E_OK;
	volatile U8*	cur_wr		= 0;
	U8*				nxt_wr		= 0;
	U32				indx		= 0;
	S32				ovrwr		= FALSE;
	TRACE_flagAttr	Attr		= {q->sh->wai_4_spc, TWF_ORW | TWF_BITCLR, 0, TMO_FEVR};
	BOOL			exit_loop	= TRUE;

	if((q != NULL) && (appl_buf != NULL))
	{
		do
		{
			exit_loop	= TRUE;
			/* PRQA: Lint Message 454: If and only if mutex_lock is success, we are unlocking. Lint is not checking */
			/* PRQA: Lint Message 454: what is the if() condition  but only checking whether unlock is called inside if() */
			/*lint -save -e454 */
			/* PRQA: Lint Message 456: If and only if mutex_lock is success, we are unlocking. Lint is not checking */
			/* PRQA: Lint Message 456: what is the if() condition  but only checking whether unlock is called inside if() */
			/*lint -save -e456 */
			rc_qlock = TRACE_obtain_q_lock(&q->sh->lock);
			if(E_OK == rc_qlock) /*SWGII-5799 Fix*/
			{
				cur_wr = GET_POINTER(q, q->sh->wr);

				/* Check for queue full condition.
				 * WR_POINTER will always stay 1 packet behind RD_POINTER
				 */
				if((((U32)cur_wr + (U32)q->sh->pkt_sz) == (U32)GET_POINTER(q, q->sh->rd))
					|| ((((U32)cur_wr + (U32)q->sh->pkt_sz) == (U32)q->end)
						&& ((U32)GET_POINTER(q, q->sh->rd) == (U32)q->start)))
				{
					if(q->sh->blck_mode == TRACE_BLOCK_MODE)
					{
						q->sh->q_full++; /*waiting tasks count*/
						TRACE_release_q_lock(&q->sh->lock);

						ret_val = TRACE_wait_flag(q->event, &Attr);

						/* Fix: SWGIIX-179 q_full is never decremented in case of BLOCK mode */
						/* SWGIIX-2341 (SWGIIX-2339)Processor watchdog reset detected */
						/* For the above fix is q_full decrementing should be also done atomically */
						rc_qlock = TRACE_obtain_q_lock(&q->sh->lock);
						if(E_OK == rc_qlock)
						{
							q->sh->q_full--;
							TRACE_release_q_lock(&q->sh->lock);
						}

						if(ret_val != E_OK)
						{
							TRACE_SYSLOG(TRACE_SYSLOG_ERROR, "In %s TRACE_wait_flag() block mode failed err=%d \n", __func__, rc_qlock);
						}
						exit_loop = FALSE;
					}
					else /*Queue in Non-Block mode*/
					{
						ovrwr = TRUE;
					}
				}
			}
			else
			{
				ret_val = rc_qlock;
			}
		} while(!exit_loop);

		/*
		 * Handling for queue full condition while no block mode is active
		 * OR caller context is a task independent portion.
		 */
		if((ovrwr == TRUE) && (rc_qlock == E_OK) ) /*Enter condition if and only if TRACE_obtain_q_lock() is success*/
		{
			if(((U32)cur_wr + (U32)q->sh->pkt_sz) == (U32)GET_POINTER(q, q->sh->rd))
			{
				if(((U32)GET_POINTER(q, q->sh->rd) + ((U32)q->sh->pkt_sz * 2)) >= (U32)q->end)
				{
					q->sh->wr = q->sh->rd;
					q->sh->rd = GET_OFFSET(q, q->start);
					q->sh->ovrwr_count += 2;
				}
				else
				{
					q->sh->wr  = q->sh->rd;
					q->sh->rd += (q->sh->pkt_sz * 2);
					q->sh->ovrwr_count += 2;
				}
			}
			else
			{
				/* @dhd3kor<28.03.2010>: Fix for SWGII-4956.
				* Trace_q_pop function enters into eternal loop.
				* Read pointer is at the start and Q wrap-around condition occurs, last packet
				* was not written due to the update of cur_wr to q->start. But pop will wait
				* for the data to be written in that location.
				*/
				/*cur_wr = q->start;*/

				q->sh->rd  = (U32)(q->sh->pkt_sz * 2);
				q->sh->ovrwr_count += 2;

			}
		}/*if((ovrwr == TRUE) && (rc_qlock == E_OK))*/

		/*
		 * Write the message into the queue
		 */
		if(E_OK == rc_qlock)
		{
			/* Update counter to indicate total number of pending packets to be read from queue. */
			q->sh->count++;

			/* Check for end of queue condition */
			if(((U32)cur_wr + (U32)q->sh->pkt_sz) >= (U32)q->end)
			{
				nxt_wr = (U8*)q->start;
			}
			else
			{
				nxt_wr = (U8*)((U32)cur_wr + (U32)q->sh->pkt_sz);
			}

			/* Calculate current index into queue */
			indx = ((U32)cur_wr - (U32)q->start)/q->sh->pkt_sz;

			/* Update flag to indicate that the current slot is yet to be filled with
			a valid packet
			*/
			q->wr_stat[indx] = TRACE_PACKET_NOT_AVAILABLE;

			/* Update WR pointer */
			q->sh->wr = GET_OFFSET(q, nxt_wr);

			TRACE_release_q_lock(&q->sh->lock);

			/* Copy user packet into queue */
			if(NULL != cur_wr) /* For removing QAC warning, cur_wr may be NULL */
			{
				memcpy((VP)cur_wr, appl_buf, q->sh->pkt_sz);
			}

			/* Update flag to indicate that a valid packet in now available */
			q->wr_stat[indx] = TRACE_PACKET_AVAILABLE;

			/* Check removed to  WR task to be  woken-up  */
			/* SWGIII-5660.Trace messages are not sent to TTFis without trigger */
			/* Earlier Threshold limit was there to trigger the write task. Because of this , if there are large number of messages
			in queue, the write task was not woken up .So check was removed */
			/*if(q->sh->count < TRACE_MIN_PKTS_IN_QUEUE)*/
			/* RTC-479458 / NCG3D-119027:-
			 * Set write task flag on TRACE_wr_task wait flag state only */
			if (g_TRACE_mgr != NULL && g_TRACE_mgr->sh != NULL && g_TRACE_mgr->sh->wr_tsk_state == 0)
			{
				ret_val = TRACE_set_flag(q->event, q->sh->dat_avl); /* Wake-up WR task */
			}
		} /*if(E_OK == rc_qlock)*/
	} /*if((q != NULL)&&(appl_buf != NULL))*/
	else
	{
		ret_val = E_FAIL;
	}
	return ret_val;
	/*lint -restore */
	/*lint -restore */
}

/**
 * Pop packet from queue.
 *
 * \param q Pointer to queue
 *
 * return \li TRACE_Q_PKTS_PENDING if more packets are to be read
 *        \li TRACE_Q_EMPTY        if no more packets exists in the queue
 *        \li                      else error returned by concrete implementation
 */
EXPORT TRACE_q_status TRACE_q_pop(TRACE_q* q)
{
	TRACE_q_status   rc       = TRACE_Q_EMPTY;
	volatile U8*     cur_rd   = NULL;
	volatile U8*     cur_wr   = NULL;
	volatile U8      status   = 0;
	volatile U8*     nxt_rd   = NULL;
	U32              indx     = 0;
	U32              counter  = 0;

	TRACE_obtain_q_lock(&q->sh->lock);
	cur_rd = GET_POINTER(q, q->sh->rd);
	cur_wr = GET_POINTER(q, q->sh->wr);
	TRACE_release_q_lock(&q->sh->lock);

	/*
	 * TODO: Return value of this API is not checking since it wont make any impact and also it increases complexity.
	 * But return value check also should be implemented in future with or without change in the implementation
	 */
	if((U32)cur_rd != (U32)cur_wr)
	{
		/* Calculate current index into queue */
		indx = ((U32)cur_rd - (U32)q->start)/q->sh->pkt_sz;

		/* Wait for availability of valid packet at current index into queue */
		do
		{
			counter++;
			if(counter > RETRY_CNT)
			{
				counter = 0;
				TRACE_sleep(WAIT_TIME_MS_10);
			}
			status = q->wr_stat[indx];
		}while(status == TRACE_PACKET_NOT_AVAILABLE);

		TRACE_obtain_q_lock(&q->sh->lock);
		/* Copy current packet from queue to a local buffer */
		memcpy((VP)q->buf, (VP)cur_rd, q->sh->pkt_sz);

		/* Update RD pointer */
		nxt_rd = (U8*)((U32)cur_rd + (U32)q->sh->pkt_sz);

		/* Check for end of queue condition */
		if((U32)nxt_rd == (U32)q->end)
		{
			q->sh->rd = GET_OFFSET(q, q->start);
		}
		else
		{
			q->sh->rd = GET_OFFSET(q, nxt_rd);
		}

		/* Update pending to-be-read packets count */
		q->sh->count--;
		q->sh->remain = q->sh->count;
		rc = TRACE_Q_PKTS_PENDING;
		if(q->sh->ovrwr_count > 0)
		{
			q->sh->ovrwr_cnt_alias = q->sh->ovrwr_count;

			/* SWGII-6425 (SWGII-6418)[b_MAIN] Reset occurred while trying to play tuner, audio files */
			TRACE_SYSLOG(TRACE_SYSLOG_ERROR, "%d messages are lost due to overwriting\n",q->sh->ovrwr_count);/*agv2kor*/

			q->sh->ovrwr_count = 0;
		}
	} /*if((U32)cur_rd != (U32)cur_wr)*/
	else
	{
		TRACE_obtain_q_lock(&q->sh->lock);
		/* Update pending to-be-read packets count to zero as queue is now empty */
		q->sh->count = 0;
		q->sh->remain = 0;
	}

	/* Wakeup application task indicating queue is now empty */
	if(q->sh->q_full > 0)
	{
		TRACE_release_q_lock(&q->sh->lock);
		TRACE_set_flag(q->event, q->sh->wai_4_spc);
	}
	else
	{
		TRACE_release_q_lock(&q->sh->lock);
	}

	return rc;
}

/**
 * This function returns the %age fill status of circular queue.
 *
 * \param q Pointer to queue
 *
 * return Queue fill status in percentage
 */
EXPORT TRACE_q_fill_stat TRACE_q_fill_status(TRACE_q* q)
{
	volatile U8*      rd   = GET_POINTER(q, q->sh->rd);
	volatile U8*      wr   = GET_POINTER(q, q->sh->wr);
	U32              tmp   = (U32)q->end - (U32)q->start;
	TRACE_q_fill_stat fill = 0;

	if((U32)wr > (U32)rd)
	{
		fill = (TRACE_q_fill_stat)((U32)wr - (U32)rd);
	}
	else if((U32)rd > (U32)wr)
	{
		fill = (TRACE_q_fill_stat)(tmp - ((U32)rd - (U32)wr));
	}
	else if((U32)wr == (U32)rd)
	{
		fill = (TRACE_q_fill_stat)tmp;
	}
	fill = (TRACE_q_fill_stat)(100 - ((fill * 100U)/tmp));
	return fill;
}

/*@dhd3kor<17.09.2012>: SWGIIX-3101(SWGIIX-3097)[b_E21_X_MC] Processor Watchdog Reset detected*/
/**
* This function releases the all the waiting task in TRACE_q_push.
*
* \param q Pointer to queue
*
* return status of the set flag function
*/
EXPORT ER TRACE_q_release_wait_tasks(TRACE_q* q)
{
	ER rc = E_OK;
	while((q->sh->q_full > 0) && (rc >= E_OK))
	{
		rc = TRACE_set_flag(q->event, q->sh->wai_4_spc);
		TRACE_sleep(TRACE_SLEEP_MS_20);
	}
	return rc;
}
